package com.ejie.ab04b.dao.nora;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.ab04b.model.nora.Municipio;
import com.gfi.constants.Constants;
import com.gfi.utils.UtilSQL;

/**
 * MunicipioDaoImpl.
 * 
 * @author GFI
 */
@Repository()
@Transactional()
public class MunicipioDaoImpl implements MunicipioDao {

	private JdbcTemplate jdbcTemplate;

	private static final Logger LOGGER = LoggerFactory
			.getLogger(MunicipioDaoImpl.class);

	private RowMapper<Municipio> rwMap = new RowMapper<Municipio>() {

		public Municipio mapRow(ResultSet resultSet, int rowNum)
				throws SQLException {

			return new Municipio(resultSet.getString("CODPROVINCIA"),
					resultSet.getString("CODMUNICIPIO"),
					resultSet.getString("DESCRIPCION"));
		}
	};

	/**
	 * Method use to set the datasource.
	 * 
	 *  dataSource
	 *            DataSource
	 *
	 * @param dataSource the new data source
	 */
	@Resource()
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Finds a single row in the Municipio table.
	 * 
	 *  municipio
	 *            Pagination
	 *  Municipio
	 *
	 * @param municipio the municipio
	 * @return the municipio
	 */
	@Transactional(readOnly = true)
	public Municipio find(Municipio municipio) {
		// SELECT - FROM
		StringBuilder query = new StringBuilder(this.getSelectFromQuery(true));

		// WHERE
		query.append(" WHERE T1.PROVINCIA_ID = ? AND T1.ID = ? ");

		StringBuilder traza = new StringBuilder("[MunicipioDaoImpl]: find ");
		traza.append(query.toString());
		traza.append("\n Parametros: ").append(municipio.getCodProvincia())
				.append(", ").append(municipio.getCodMunicipio());
		MunicipioDaoImpl.LOGGER.info(traza.toString());

		List<Municipio> municipioList = this.jdbcTemplate.query(
				query.toString(), this.rwMap, municipio.getCodProvincia(),
				municipio.getCodMunicipio());
		return (Municipio) DataAccessUtils.uniqueResult(municipioList);
	}

	/**
	 * Finds a List of rows in the Municipio table.
	 * 
	 *  municipio
	 *            Municipio
	 *  List
	 *
	 * @param municipio the municipio
	 * @return the list
	 */
	@Transactional(readOnly = true)
	public List<Municipio> findAll(Municipio municipio) {
		// SELECT - FROM
		StringBuilder query = new StringBuilder(this.getSelectFromQuery(true));

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereMap(municipio);
		StringBuilder where = new StringBuilder(" WHERE 1=1 ");
		where.append(mapaWhere.get("query"));
		query.append(where);
		query.append(" ORDER BY DS_O");

		List<?> params = (List<?>) mapaWhere.get("params");

		return (List<Municipio>) this.jdbcTemplate.query(query.toString(),
				this.rwMap, params.toArray());
	}

	/**
	 * Counts rows in the Municipio table.
	 * 
	 *  municipio
	 *            Municipio
	 *  Long
	 *
	 * @param municipio the municipio
	 * @return the long
	 */
	@Transactional(readOnly = true)
	public Long findAllCount(Municipio municipio) {
		StringBuilder query = new StringBuilder(
				"SELECT COUNT(1) FROM MUNICIPIO T1 ");

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereMap(municipio);
		StringBuilder where = new StringBuilder(" WHERE 1=1 ");
		where.append(mapaWhere.get("query"));
		query.append(where);

		List<?> params = (List<?>) mapaWhere.get("params");

		return this.jdbcTemplate.queryForLong(query.toString(),
				params.toArray());
	}

	/*
	 * MÉTODOS PRIVADOS
	 */

	/**
	 * Devuelve la Select con todos los campos. Incluye el From si se indica.
	 * 
	 *  includeFrom
	 *            boolean
	 *  String
	 *
	 * @param includeFrom the include from
	 * @return the select from query
	 */
	private String getSelectFromQuery(boolean includeFrom) {
		StringBuilder query = new StringBuilder(
				"SELECT T1.PROVINCIA_ID CODPROVINCIA, T1.ID CODMUNICIPIO, T1.DS_O DESCRIPCION ");
		if (includeFrom) {
			query.append(" FROM MUNICIPIO T1 ");
		}
		return query.toString();
	}

	/**
	 * Returns a map with the needed value to create the conditions to filter by
	 * the Municipio entity
	 * 
	 *  municipio
	 *            Municipio Bean with the criteria values to filter by.
	 *  Map created with two keys key query stores the sql query syntax
	 *         key params stores the parameter values to be used in the
	 *         condition sentence.
	 *
	 * @param municipio the municipio
	 * @return the where map
	 */
	private Map<String, ?> getWhereMap(Municipio municipio) {

		StringBuilder where = new StringBuilder(Constants.STRING_BUILDER_INIT);
		List<Object> params = new ArrayList<Object>();

		if (municipio != null) {
			if (municipio.getCodMunicipio() != null) {
				UtilSQL.createFilterSQL(where, params, "T1.ID",
						municipio.getCodMunicipio());
			}
			if (municipio.getCodProvincia() != null) {
				UtilSQL.createFilterSQL(where, params, "T1.PROVINCIA_ID",
						municipio.getCodProvincia());
			}
			if (municipio.getDescripcion() != null) {
				UtilSQL.createFilterLikeSQL(where, params, "T1.DS_O",
						municipio.getDescripcion(), false);
			}
		}

		Map<String, Object> mapWhere = new HashMap<String, Object>();
		mapWhere.put("query", where);
		mapWhere.put("params", params);

		return mapWhere;
	}

	/**
	 * Finds a List of rows in the Municipio table.
	 * 
	 *  municipio
	 *            Municipio
	 *  List
	 *
	 * @param municipio the municipio
	 * @return the municipio
	 */
	@Transactional(readOnly = true)
	public Municipio findAllFirst(Municipio municipio) {
		StringBuilder query = new StringBuilder(
				"SELECT T1.PROVINCIA_ID CODPROVINCIA, T1.ID CODMUNICIPIO, T1.DS_O DESCRIPCION  FROM MUNICIPIO T1 WHERE "
						+ "T1.PROVINCIA_ID=?  and UPPER(translate(T1.DS_O, 'áéíóúÁÉÍÓÚ', 'aeiouAEIOU')) LIKE '%'||UPPER(translate(?, 'áéíóúÁÉÍÓÚ', 'aeiouAEIOU'))||'%' and rownum=1 ");

		query.append(" ORDER BY T1.DS_O");

		List<Municipio> municipioList = this.jdbcTemplate.query(
				query.toString(), this.rwMap, municipio.getCodProvincia(),
				municipio.getDescripcion());

		return (Municipio) DataAccessUtils.uniqueResult(municipioList);
	}

}